/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_19; import java.io.*;
import java.util.*; import djbuch.kapitel_06.DemoTools;
import lotus.domino.*; import org.apache.log4j.*;

/** Ausgabe Klasse fr Debug-Informationen
 * 
 * Kapselt einen Log4J Logger
 * Ausgabe Loglevel wird durch die Konstante DO_NOT_LOG_BEYOND_THIS beschrnkt.
 * Log4J ist selbstkonfigurierend. Das zugehrige Config File wird (sofern 
 * noch nicht vorhanden) in <i>user.dir</i>/java angelegt
 * und kann dort dann auch modifiziert werden.
 * Vorbereitet ist ein NotesLogAppender, der in eine Notes Datenbank logged.
 * Diesr muss im Config File aktiviert werden.
 * Stellt einige hilfreiche Ausgabe Methoden fr z.B. Enumeration, Vector, Exception
 * oder Document bereit
 * Falls getTraceStatus()==true, dann wird jeder ausgabe ein TracePfad vorangestellt,
 * so dass jeder Ausgabe die zugehrige Methode angesehen werden kann.
 * 
 * @author Thomas Ekert
 */
//TSA-JAVA0136
public class DJLog
{
	
	private static final int NOT_LOADED = -1;
	public static final int ERROR = 0;
	public static final int WARN = 1;
	public static final int INFO = 2;
	public static final int DEBUG = 3;
	public static final int DEBUG2 = 4;
	public static final int DEBUG3 = 5;

	private static final int CARRIAGE_RETURN = 13;
	private static final int LINE_FEED = 10;
	//initiale Stacktiefe, die ausgeblendet werden soll.
	private static final int INITIAL_STACK_SIZE = 7; 
	private static final int DO_NOT_LOG_BEYOND_THIS = DJLog.DEBUG;

	int currentLogLevel = NOT_LOADED;
	private static boolean showTrace = true;
	
	/**
	 * @see write (int, String, boolean)
	 * gibt einen Trace nur dann aus, wenn fr die Klasse ein Trace zugelassen ist.
	 * @param logLevel
	 * @param msg
	 */
	public static void write (int logLevel, String msg) {
		write (logLevel, msg, !showTrace );
	}
	
	/**
	 * Schreibt eine Stringnachricht ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult. 
	 * @param logLevel - Level auf dem gelogged werden soll
	 * @param msg - Log Nachricht
	 * @param kurz - falls kurz==true, wird kein Trace (Anzeige der aufrufenden Klasse)
	 * ausgegeben.
	 */	 public static void write(int logLevel, String msg, boolean kurz)
	{
		if (logLevel > DO_NOT_LOG_BEYOND_THIS) {
			return;
		}


		if (logLevel <= DJLog.singleton.currentLogLevel) {
			/* mehrere sequentielle Aufrufe von System.out.print{,ln} resultieren
			 * in jeweils einer neuen Zeile.
			 */
			 int t = callStackSize() - INITIAL_STACK_SIZE;
			 if (t < 1) { t = 0; }
			 if (kurz) {
			 	singleton.logger.log (internalToLog4jLevel(logLevel), msg);
			 } else {
			 	singleton.logger.log (internalToLog4jLevel(logLevel), logSymbol (logLevel) +" " +  times(". ", t) + lastCaller("DJLog") + ": " + msg);
			 }
		}
	}

	/**
 	 * Schreibt eine Stringnachricht ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult. 
	 * @see write (int, String)
	 * @param logLevel
	 * @param msg
	 */
	public static void writeString(int logLevel, String msg)
	{
		write(logLevel, msg);
	}

	/**
	 * Schreibt eine Schlssel-Wert-Liste ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult.
	 * @param logLevel
	 * @param keys
	 * @param values
	 */
	public static void writePairs(int logLevel, Vector keys, Vector values)
	{
		StringWriter sw = new StringWriter();
		PrintWriter ps = new PrintWriter(sw);
		ps.println("" + keys.size() + " Schlssel und " + values.size() + " Werte:");
		for (int i = 0; i < keys.size() || i < values.size(); i++) {
			ps.print((i < keys.size())? keys.elementAt(i): "(null)");
			ps.print("=");
			ps.println((i < values.size())? values.elementAt(i): "(null)");
		}
		write(logLevel, sw.toString());
	}

	/**
 	 * Schreibt eine Exception mit StackTrace ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult.
	 * @param logLevel
	 * @param e
	 */
	public static void writeException(int logLevel, Throwable e)
	{
		if (e == null) {
			write (logLevel, "unknown exeption");
		}
		else {
			StringWriter sw = new StringWriter();
			e.printStackTrace(new PrintWriter(sw));
			if (e instanceof NotesException) {
				NotesException ne = (NotesException) e;
				write(logLevel, "NotesException #" + ne.id + ": " + ne.text);
			}
			try {
				nicewrite(logLevel, "\n" + sw.toString());
			} catch (IOException ex) {
				System.err.println("FATAL: DJLog.writeException: " + ex);
			}
		}
	}

	/**
	 * Schreibt die Feldinhalte eines Notesdokumentes ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult. 
	 * @param doc
	 */
	public static void writeDocument(Document doc)
	{
		StringWriter sw = new StringWriter();
		try {
			DemoTools.dumpDoc(doc, new PrintWriter(sw));
			write(INFO, "Document: ");
			nicewrite (INFO,sw.toString());
		} catch (NotesException e) {
			System.err.println("FATAL: Notes Exception in DJLog.writeDocument: " + e.toString() + e.text);
		} catch (IOException e) {
		System.err.println("FATAL: DJLog.writeDocument: " + e.toString());
	}
	}

	/* setzt lange strings so um, dass write eine saubere ausgabe auf die (Domino) Konsole macht */
	private static void nicewrite (int logLevel, String message) throws IOException {
		String sLine = "";			
		StringReader reader= new StringReader (message);
		for (int s =  reader.read();s>0; s =  reader.read())
			{
				if (s > 0 && s != CARRIAGE_RETURN && s!= LINE_FEED) {
					if (s!=0) { sLine += (char) s; }
				} else {
					if (sLine != null && !sLine.equals( "")) { write (logLevel,sLine, true); }
					sLine = "";
				}
			}
	}

	/**
	 * Schreibt eine Exception mit Loglevel <code>ERROR</code> ins Log,
	 * wenn das im Profildokument eingestellte Loglevel es zult. 
	 * @param e
	 */
	public static void writeException(Throwable e)
	{
		writeException(ERROR, e);
	}

	/**
	 * Schreibt eine Aufzhlung ins Log, wenn das im Profildokument
	 * eingestellte Loglevel es zult.
	 * @param logLevel
	 * @param e - auszugebende Enumeration
	 */
	public static void writeEnumeration(int logLevel, Enumeration e)
	{
		if (e == null) {
			write(logLevel, "writeEnumeration: null");
			return;
		}
		write(logLevel, e.getClass().getName());
		for (int i = 0; e.hasMoreElements(); i++) {
			write(logLevel, "" + i + ": " + e.nextElement());
		}
	}
	
	/**
	 * Schreibt einen Vector ins Log
	 * @param logLevel
	 * @param v
	 */
	public static void writeVector(int logLevel, Vector v)
	{
		if (v == null) {
			write(logLevel, "writeVector: null");
			return;
		}
		write(logLevel, v.getClass().getName());
		for (int i = 0, k = v.size(); i<k; i++) {
			try {
				write(logLevel, "" + i + ": " + v.elementAt(i).toString());
			} //TSA-Java0166
				catch (Exception e) {
				write (logLevel, "" + i + ": Exception " + e.toString());
			}
		}
	}
	
	/**
	 * ndert das Loglevel.
	 * @param newLogLevel
	 */
	public static void setLogLevel(int newLogLevel) {
		singleton.currentLogLevel = newLogLevel;
	}
	/**
	 * Gibt das Loglevel aus.
	 * @return
	 */
	public static int getLogLevel() {
		return singleton.currentLogLevel;
	}
	
	/**
	 * legt fest, ob bei der Ausgabe ein Stacktrace angegeben werden soll.
	 * @param newTraceStatus - true oder false
	 */
	public static void setTraceStatus (boolean newTraceStatus) {
		showTrace = newTraceStatus;
	}
	
	/** 
	 * zeigt an, ob zur Zeit ein Stacktrace beim Loggen ausgegeben wird oder nicht
	 * @return
	 */
	public static boolean getTraceStatus () {
		return showTrace;
	}
	
	/**
	 * Schreibt <i>msg</i> mit dem Level ERROR in die Logdatei.
	 * @param msg
	 */
	public static void error(String msg) {
		write(ERROR, msg); 
	}
	
	/**
	 * Schreibt <i>msg</i> mit dem Level WARN in die Logdatei.
	 * @param msg
	 */
	public static void warn(String msg) {
		write(WARN, msg);
	}
	
	/**
	 * Schreibt <i>msg</i> mit dem Level INFO in die Logdatei.
	 * @param msg
	 */
	public static void info(String msg) {
		write(INFO, msg);
	}
	
	/**
	 * Schreibt <i>msg</i> mit dem Level DEBUG in die Logdatei.
	 * @param msg
	 */
	public static void debug(String msg) {
		write(DEBUG, msg);
	}
	
	/**********************************************************************
	 * verschiedene Hilfs Methoden zum Handling des internen Log4J Logger
	 * und der Ausgabeformatierung
	 **********************************************************************/
	
	/**
	 * legt den initialen LogLevel Fest
	 */
	private void loadLogLevel()
	{
		currentLogLevel = INFO; /* Standard-Wert */
		//Optional hier die Mglichkeit einfgen, den Wert aus 
		//einer Properties Datei zu laden...
		//Dieser Wert wird dann bei jeder initialisierung gezogen
		if (currentLogLevel > WARN) {
			System.out.println ("LogLevel=" + currentLogLevel);
		}
	}
	
	/** 
	 * Konvertiert die Konstanten aus DJLog in Level Objekte aus Log4J
	 * @param source
	 * @return
	 */
	public static final Level internalToLog4jLevel (int source) {
		switch (source) {
			case (ERROR) : return Level.FATAL;
			case (WARN) : return Level.WARN; 
			case (INFO) : return Level.INFO; 
			case (DEBUG) : return Level.DEBUG; 
			case (DEBUG2) : return Level.ALL; 
			case (DEBUG3) : return Level.ALL; 
			default : return Level.ALL; 
		}
	}
	
	/**
	 * Liefert den Namen einer aufrufenden Methode oder deren Aufrufer
	 * @param callerID - gibt an, wieviele Aufrufe oberhalb der aufrufenden Methode
	 * ausgegeben werden sollen. 0 liefert den Namen der aufrufenden Methode, 1 den
	 * Namen des bergeordneten Aufrufers usw.
	 * @return - Name der Aufrufenden Methode
	 */
	private static String getCaller(int callerID) {
		int stack_base = callerID+4;
		// +1 to ignore "Throwable" line, +1 to ignore this method
		StringWriter sw = new StringWriter();
		(new Throwable()).printStackTrace(new PrintWriter(sw));
		String trace = sw.toString();
		int linestart = -1;
		for (int i = 0; i < stack_base; i++) {
			linestart = trace.indexOf("\n", linestart + 1);
		}
		return trace.substring(linestart + 5, trace.indexOf("(", linestart + 5));
	}

	/**
	 * Ermittelt den Namen der aufrufenden Funktion.
	 */
	public static String lastCaller(String className)
	{
		String lastCaller = "";
		for (int i = 1; true; i++) {
			lastCaller = getCaller(i);
			if (lastCaller == null) {
				return "keine aufrufende Methode";
			}
			if (lastCaller.indexOf(className + ".") < 0) {
				return lastCaller;
			}
		}
	}
	
	/**
	 * Ermittelt die Tiefe des Aufrufstacks.
	 */
	private static int callStackSize()
	{
        StringWriter sw = new StringWriter();
        (new Throwable()).printStackTrace(new PrintWriter(sw));
        String trace = sw.toString();
        int size = 0;
        for (int nl = 0; nl != -1; nl = trace.indexOf("\n", nl + 1)) {
        	size++;
        }
        return size;
    }

	/**
	 * Erzeugt Symbole, um im Log die verschiedenen Level unterscheiden zu knnen.
	 * @param i
	 * @return
	 */
	public static String logSymbol (int i) {
		switch (i) {
			case ERROR:
				return "[!!!]";
			case WARN:
				return "[ ! ]";
			case INFO:
				return "[ - ]";
			default:
				return "     ";
		}
	}

    private static String times(String s, int n)
    {
		StringBuffer b = new StringBuffer();
		for (int i = 0; i < n; i++) {
			b.append(s);
		}
		return b.toString();
	}

    /**
     * Default Konfiguration fr log4j
     * Diese wird (falls noch nicht vorhanden) im Userverzeichnis (<<user.dir>>/java) 
     * der aufrufenden Person gespeichert und kann dort modifiziert werden.
     * @see setBasePath (String)
     */
	private static final String defaultProperties =  "#\n"
		 + "# Automatisch durch DJLog generiertes property file fr log4j\n"
		 + "# \n"
		 + "\n"
		 + "log4j.rootLogger=@@LEVEL@@, STDOUTLOGGER\n"
		 + "#log4j.rootLogger=@@LEVEL@@, ROLL, NOTESLOGGER\n"
		 + "#log4j.rootLogger=@@LEVEL@@, ROLL, STDOUTLOGGER\n"
		 + "\n"
		 + "#################################################################\n"
		 + "### Definition for Stdout logger\n"
		 + "#################################################################\n"
		 + "\n"
		 + "log4j.appender.STDOUTLOGGER=org.apache.log4j.ConsoleAppender\n"
		 + "log4j.appender.STDOUTLOGGER.layout=org.apache.log4j.PatternLayout\n"
		 + "\n"
		 + "# Pattern to output the caller's file name and line number.\n"
		 + "log4j.appender.STDOUTLOGGER.layout.ConversionPattern=%d [%t] %-5p %c - %m%n\n"
		 + "\n"
		 + "\n"
		 + "#################################################################\n"
		 + "### Definition for Notes logger - writes to notes db\n"
		 + "#################################################################\n"
		 + "\n"
		 + "log4j.appender.NOTESLOGGER=djbuch.kapitel_19.NotesLogAppender\n"
		 + "log4j.appender.NOTESLOGGER.layout=org.apache.log4j.PatternLayout\n"
		 + "\n"
		 + "# Pattern to output the caller's file name and line number.\n"
		 + "log4j.appender.NOTESLOGGER.layout.ConversionPattern=%d [%t] %-5p %c - %m%n\n"
		 + "\n"
		 + "log4j.appender.NOTESLOGGER.File=@@FILENAMENOTES@@\n"
		 + "\n"
		 + "#Max Size of temporary files. Dont choose to large value, because these files will be converted to notes documents.\n"
		 + "log4j.appender.NOTESLOGGER.maxFileSize=5KB\n"
		 + "# Keep 10 backup files\n"
		 + "log4j.appender.NOTESLOGGER.maxBackupIndex=10\n"
		 + "\n"
		 + "# Name of target Database\n"
		 + "log4j.appender.NOTESLOGGER.notesDb=djbuch/djbuch.nsf\n"
		 + "\n"
		 + "# Domino Server Name (e.g. ServerName/Organisation)\n"
		 + "#log4j.appender.NOTESLOGGER.dominoServer=Java/DJBUCH\n"
		 + "\n"
		 + "# Password for local or IIOP Session - can be left empty on server\n"
		 + "#log4j.appender.NOTESLOGGER.password=secret\n"
		 + "\n"
		 + "#\n"
		 + "# Optional - Fill iiop Variables to establishe IIOP Connection instead of local connection\n"
		 + "# When using this: Be carefull to be shure, that the iiopServer Domain is the same Server as dominoServer\n"
		 + "# Otherwise you will get security exceptions\n"
		 + "#\n"
		 + "# IIOP User\n"
		 + "#log4j.appender.NOTESLOGGER.iiopUser=user\n"
		 + "\n"
		 + "# Domino IIOP Server Domainname (e.g. my.Server.com)\n"
		 + "#log4j.appender.NOTESLOGGER.iiopServer=127.0.0.1\n"
		 + "\n"
		 + "#################################################################\n"
		 + "### Definition for Rolling File Appender logger\n"
		 + "#################################################################\n"
		 + "\n"
		 + "log4j.appender.ROLL=org.apache.log4j.RollingFileAppender\n"
		 + "log4j.appender.ROLL.File=@@FILENAME@@\n"
		 + "\n"
		 + "log4j.appender.ROLL.maxFileSize=1MB\n"
		 + "# Keep 10 backup files\n"
		 + "log4j.appender.ROLL.maxBackupIndex=10\n"
		 + "\n"
		 + "log4j.appender.ROLL.layout=org.apache.log4j.PatternLayout\n"
		 + "log4j.appender.ROLL.layout.ConversionPattern=%d [%t] %-5p %c - %m%n\n"
		 + "\n";
	 
	/**
	 * log4j mit einer Config-Datei einrichten.
	 * @param filename
	 */

	protected void setupLog4j(String filename) {
		if (DemoTools.exists(basePath + filename)) {
			System.out.println ("Loading Log4j Config from file: "+ basePath+filename);
			PropertyConfigurator.configure (basePath+filename);
		} else {
			try {
				System.out.println ("Generating NEW Log4j Default Config in file: "+ basePath+filename);
				DemoTools.makeDir(basePath);
				FileWriter fw = new FileWriter(basePath+filename);
				fw.write(insertMaxLevel (defaultProperties));
				fw.flush();
				fw.close();
				 //TSA-JAVA0174
				fw = null;
				PropertyConfigurator.configure (basePath+filename);
			} catch (IOException e) {
				BasicConfigurator.configure();
			}
		}
	}
	
	/**
	 * Abhngig von der Konstanten DO_NOT_LOG_BEYOND_THIS dieser Klasse,
	 * wird das default-log4j-config-file geschrieben.
	 * @param input
	 * @return
	 */

	private String insertMaxLevel(String input) {
		String levelName = "INFO";
		//DEBUG, INFO, WARN, ERROR and FATAL 
		switch (DO_NOT_LOG_BEYOND_THIS) {
			case (ERROR) : levelName = "FATAL"; break;
			case (WARN) : levelName = "WARN"; break;
			case (INFO) : levelName = "INFO"; break;
			case (DEBUG) : levelName = "DEBUG"; break;
			case (DEBUG2) : levelName = "ALL"; break;
			case (DEBUG3) : levelName = "ALL"; break;
			default: levelName = "ALL"; break;
		}
		return DemoTools
				.replaceSubstring(DemoTools.replaceSubstring(DemoTools.replaceSubstring(input, "@@LEVEL@@",
						levelName), "@@FILENAME@@", basePath + "djbuch.log"), "@@FILENAMENOTES@@", basePath
						+ "djbuchnotes.log");
	}
	
	/**
	 * Der basePath der Log Klasse ist per Default das Verzeichnis <<user.dir>>/java
	 * Es kann mit setBasePath verndert werden.
	 * In diesem Pfad werden logs und die log4j.conf gespeichert.
	 * @param path
	 */
	public static final void setBasePath (String path) {
		singleton.basePath=path;
	}

	/**
	 * @return
	 */
	public static String getBasePath() {
		return singleton.basePath;
	} 
	
	private static DJLog singleton = new DJLog();
	private  Logger logger = null;
	private String basePath = DemoTools.replaceSubstring(System.getProperty("user.dir"),"\\","/") + "/java/";
	private String configFilename = "log4j.conf";
	    
	private DJLog () {
		loadLogLevel();
		logger = Logger.getLogger("DJLog");
		setupLog4j(configFilename);
	}

}
